/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.biometricprompt.cipher;

import android.app.KeyguardManager;
import android.content.Context;
import android.content.DialogInterface;
import android.os.CancellationSignal;
import android.os.Handler;
import android.security.keystore.KeyGenParameterSpec;
import android.security.keystore.KeyInfo;
import android.security.keystore.KeyPermanentlyInvalidatedException;
import android.security.keystore.KeyProperties;

import androidx.biometric.BiometricPrompt;
import androidx.fragment.app.FragmentActivity;

import java.io.IOException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.concurrent.Executor;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;

public class BiometricAuthentication {
    private static final String TAG = "BioAuth";

    private static final String KEY_NAME = "KeyForFingerprintAuthentication";
    private static final String PROVIDER_NAME = "AndroidKeyStore";
    private androidx.biometric.BiometricPrompt mBiometricPrompt;
    private androidx.biometric.BiometricPrompt.PromptInfo mPromptInfo;
    private CancellationSignal mCancellationSignal;
    private KeyStore mKeyStore;
    private KeyGenerator mKeyGenerator;
    private Cipher mCipher;


    public BiometricAuthentication(FragmentActivity context, final androidx.biometric.BiometricPrompt.AuthenticationCallback callback) {
        // Callback which receives the result of biometric authentication
        androidx.biometric.BiometricPrompt.AuthenticationCallback hook =
                new androidx.biometric.BiometricPrompt.AuthenticationCallback() {
            @Override
            public void onAuthenticationError(int errorCode,
                                              CharSequence errString) {
                android.util.Log.e(TAG, "onAuthenticationError");
                if (callback != null) {
                    callback.onAuthenticationError(errorCode, errString);
                }
                reset();
            }

            @Override
            public void onAuthenticationSucceeded(androidx.biometric.BiometricPrompt.AuthenticationResult result) {
                android.util.Log.e(TAG, "onAuthenticationSuccess");
                if (callback != null) {
                    callback.onAuthenticationSucceeded(result);
                }
                reset();
            }

            @Override
            public void onAuthenticationFailed() {
                android.util.Log.e(TAG, "onAuthenticationFailed");
                if (callback != null) {
                    callback.onAuthenticationFailed();
                }
            }
        };

        final Handler mHandler = new Handler(context.getMainLooper());
        final Executor mExecutor = new Executor() {
            @Override
            public void execute(Runnable runnable) {
                mHandler.post(runnable);
            }
        };

        mBiometricPrompt =
            new androidx.biometric.BiometricPrompt(context, mExecutor, hook);
        final androidx.biometric.BiometricPrompt.PromptInfo.Builder builder =
            new androidx.biometric.BiometricPrompt.PromptInfo.Builder()
                .setTitle("Please Authenticate")
                .setNegativeButtonText("Cancel");
        mPromptInfo = builder.build();
        reset();
    }

    public boolean startAuthentication() {
        if (!generateAndStoreKey())
            return false;

        if (!initializeCipherObject())
            return false;

        androidx.biometric.BiometricPrompt.CryptoObject cryptoObject =
            new BiometricPrompt.CryptoObject(mCipher);

        //  Process biometric authentication
        android.util.Log.e(TAG, "Starting authentication");
        mBiometricPrompt.authenticate(mPromptInfo, cryptoObject);
        return true;
    }

    private void reset() {
        try {
            // *** POINT 2 ** Obtain an instance from the
            // “AndroidKeyStore” Provider.
            mKeyStore = KeyStore.getInstance(PROVIDER_NAME);
            mKeyGenerator =
                KeyGenerator.getInstance(KeyProperties.KEY_ALGORITHM_AES,
                                         PROVIDER_NAME);
            mCipher =
                Cipher.getInstance(KeyProperties.KEY_ALGORITHM_AES
                                   + "/" + KeyProperties.BLOCK_MODE_CBC
                                   + "/" + KeyProperties.ENCRYPTION_PADDING_PKCS7);
        } catch (KeyStoreException | NoSuchPaddingException
                 | NoSuchAlgorithmException | NoSuchProviderException e) {
            throw new RuntimeException("failed to get cipher instances", e);
        }
        mCancellationSignal = null;
    }

    private boolean generateAndStoreKey() {
        try {
            mKeyStore.load(null);
            if (mKeyStore.containsAlias(KEY_NAME))
                mKeyStore.deleteEntry(KEY_NAME);
            mKeyGenerator.init(
                // *** POINT 4 *** When creating (registering) keys,
                // use an encryption algorithm that is not vulnerable
                // (meets standards)
                new KeyGenParameterSpec.Builder(KEY_NAME, KeyProperties.PURPOSE_ENCRYPT)
                    .setBlockModes(KeyProperties.BLOCK_MODE_CBC)
                    .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_PKCS7)
                    // *** POINT 5 *** When creating (registering) keys, enable
                    // requests for user (fingerprint) authentication (do not
                    // specify the duration over which authentication is enabled)
                    .setUserAuthenticationRequired(true)
                    .build());
            // Generate a key and store it to Keystore(AndroidKeyStore)
            mKeyGenerator.generateKey();
            return true;
        } catch (IllegalStateException e) {
            return false;
        } catch (NoSuchAlgorithmException | InvalidAlgorithmParameterException
                 | CertificateException | KeyStoreException | IOException e) {
            android.util.Log.e(TAG, "key generation failed: " + e.getMessage());
            throw new RuntimeException("failed to generate a key", e);
        }
    }

    private boolean initializeCipherObject() {
        try {
            mKeyStore.load(null);
            SecretKey key = (SecretKey) mKeyStore.getKey(KEY_NAME, null);
            SecretKeyFactory factory =
                SecretKeyFactory.getInstance(KeyProperties.KEY_ALGORITHM_AES,
                                             PROVIDER_NAME);
            KeyInfo info = (KeyInfo) factory.getKeySpec(key, KeyInfo.class);

            mCipher.init(Cipher.ENCRYPT_MODE, key);
            return true;
        } catch (KeyPermanentlyInvalidatedException e) {
            // *** POINT 6 *** Design your application on the assumption that
            // the status of fingerprint registration will change between
            // when keys are created and when keys are used
            return false;
        } catch (KeyStoreException | CertificateException
                 | UnrecoverableKeyException | IOException
                 | NoSuchAlgorithmException | InvalidKeySpecException
                 | NoSuchProviderException | InvalidKeyException e) {
            android.util.Log.e(TAG, "failed to init Cipher: " + e.getMessage());
            throw new RuntimeException("failed to init Cipher", e);
        }
    }
}
